<?php
/**
 * Plugin Name: WP All Import – Gallery Sync (Direct Hook)
 * Description: Variante directe qui synchronise les images dans JetEngine sans passer par le hook shutdown.
 * Version: 2.0
 * Author: dtrabelsi
 */

defined('ABSPATH') || exit;

define('GALLERY_SYNC_LOG_PATH', plugin_dir_path(__FILE__) . 'log-wpai.txt');

// Hook unique qui fait tout directement
add_action('pmxi_saved_post', function($post_id, $xml_node, $import_id) {
    $time = date('Y-m-d H:i:s');

    // Log hook déclenché
    file_put_contents(GALLERY_SYNC_LOG_PATH, "$time 🔥 Hook déclenché pour le post $post_id (import_id: $import_id)\n", FILE_APPEND);

    $field_name = get_option('gallery_sync_field', '_photos');
    file_put_contents(GALLERY_SYNC_LOG_PATH, "$time 💬 Champ utilisé : $field_name\n", FILE_APPEND);

    $start_time = microtime(true);

    // Récupère les images attachées au post
    $attachments = get_posts([
        'post_type'      => 'attachment',
        'post_mime_type' => 'image',
        'numberposts'    => -1,
        'post_status'    => 'inherit',
        'orderby'        => 'menu_order',
        'order'          => 'ASC',
        'post_parent'    => $post_id
    ]);

    $ids = wp_list_pluck($attachments, 'ID');

    delete_post_meta($post_id, $field_name);
    update_post_meta($post_id, $field_name, $ids);

    $duration = round(microtime(true) - $start_time, 4);

    file_put_contents(
        GALLERY_SYNC_LOG_PATH,
        "$time 🔁 FINAL $post_id → $field_name: " . implode(',', $ids) . " (⏱ {$duration}s)\n",
        FILE_APPEND
    );
}, 10, 3);

// Interface admin
add_action('admin_menu', function() {
    add_menu_page(
        'Gallery Sync Settings',
        'Gallery Sync',
        'manage_options',
        'gallery-sync-settings',
        'gallery_sync_admin_page',
        'dashicons-images-alt2',
        100
    );
});

add_action('wp_ajax_gallery_sync_get_logs', function() {
    if (file_exists(GALLERY_SYNC_LOG_PATH)) {
        echo file_get_contents(GALLERY_SYNC_LOG_PATH);
    } else {
        echo "Aucun log trouvé.";
    }
    wp_die();
});

function gallery_sync_admin_page() {
    echo '<div class="wrap">';
    echo '<h1>WP All Import – Gallery Sync</h1>';
    echo '<p>Ce plugin attache les images importées aux biens et met à jour le champ JetEngine <code>_photos</code> (modifiable ci-dessous).</p>';

    // Sauvegarde du champ personnalisé
    if (isset($_POST['field_name'])) {
        update_option('gallery_sync_field', sanitize_text_field($_POST['field_name']));
        echo '<div class="updated"><p>Champ enregistré ✅</p></div>';
    }

    $current_field = get_option('gallery_sync_field', '_photos');

    echo '<form method="post" style="margin-bottom:20px;">
        <label for="field_name"><strong>Nom du champ JetEngine</strong></label><br>
        <input type="text" id="field_name" name="field_name" value="' . esc_attr($current_field) . '" style="width:300px;">
        <button type="submit" class="button button-primary" style="margin-left:10px;">Enregistrer</button>
    </form>';

    echo '<form method="post" style="margin-bottom:10px;"><button name="clear_log" class="button">🧹 Vider les logs</button></form>';

    if (isset($_POST['clear_log'])) {
        if (file_exists(GALLERY_SYNC_LOG_PATH)) {
            file_put_contents(GALLERY_SYNC_LOG_PATH, "");
        }
    }

    echo '<h2>Logs en temps réel</h2>';
    echo '<div id="log-output" style="background:#fff;border:1px solid #ccc;padding:10px;margin-top:10px;height:400px;overflow:auto;font-family:monospace;"></div>';

    echo '<script>
        function fetchLogs() {
            fetch("' . admin_url('admin-ajax.php?action=gallery_sync_get_logs') . '")
                .then(response => response.text())
                .then(data => {
                    const logBox = document.getElementById("log-output");
                    logBox.innerHTML = data
                        .replace(/\n/g, "<br>")
                        .replace(/ 🔥/g, " <span style=\"color:orange;\">🔥</span>")
                        .replace(/ 🔁/g, " <span style=\"color:green;\">🔁</span>")
                        .replace(/ 💤/g, " <span style=\"color:gray;\">💤</span>");
                    logBox.scrollTop = logBox.scrollHeight;
                });
        }
        setInterval(fetchLogs, 5000);
        fetchLogs();
    </script>';

    echo '</div>';
}
